// Macro Guard
#ifndef INPUT_FEATDIST_H
#define INPUT_FEATDIST_H

// Files included
#include "Assertion.hpp"
#include "FeatBedReader.hpp"
#include "FeatureDistanceDefns.hpp"


namespace FeatDist {


//=======
// Input
//=======
struct Input {
  // Constructor
  Input(int argc, char** argv)
                : allowCompleteOverlaps_(true),
                  strandCompare_(false), closestCoord_(false),
                  current_(0) {

    try {
      typedef InputError IE;
      typedef HelpException HE;
      Assert<HE>(argc > 1, "Help!");
      Assert<IE>(argc > 2, "Not enough input files");
      int start = 1;
      bool onlyOne = true;
      for ( int i = start; i < argc; ++i ) {
        std::string next(argv[i]);
        if ( next == "-p" ) // strand polarity
          strandCompare_ = true;
        else if ( next == "-n" ) // no exact overlaps
          allowCompleteOverlaps_ = false;
        else if ( next == "-c" ) // include closest coordinate
          closestCoord_ = true;
        else if ( next == "-" ) {
          Assert<IE>(onlyOne, "Too many '-'");
          allFiles_.push_back(USESTDIN);
          onlyOne = !onlyOne;
        }
        else {
          std::ifstream check(argv[i]);
          Assert<IE>(check, "Cannot find " + std::string(argv[i]));
          allFiles_.push_back(argv[i]);
        }
      } // for
    } catch(HelpException& he) {
      HelpException toThrow(usage());
      throw(toThrow);
    } catch(InputError& ie) {
      std::string msg = usage();
      msg += "\n\nBad Input\n";
      msg += ie.GetMessage();
      InputError toThrow(msg);
      throw(toThrow);
    }
  }

  // Public methods
  bool AllowCompleteOverlaps() const {
    return(allowCompleteOverlaps_);
  }
  bool IncludeClosestID() const {
    return(closestCoord_);
  }
  BedReader* NextBed() {
    bool isOK = current_ < allFiles_.size();
    Assert<ProgramError>(isOK, "BedReader Creation Failure");
    if ( allFiles_[current_] != USESTDIN ) {
      /* streams_'s elements must persists until Input dies */
      streams_.push_back(new std::ifstream(allFiles_[current_].c_str()));
      std::istream_iterator<ByLine> k(*(streams_.back()));
      ++current_;
      return(new BedReader(k));
    }
    ++current_;
    std::istream_iterator<ByLine> k(std::cin);
    return(new BedReader(k));
  }
  int NumberFiles() const {
    return(allFiles_.size());
  }
  bool StrandCompare() const {
    return(strandCompare_);
  }
  ~Input() {
    std::size_t sz = streams_.size();
    for ( std::size_t i = 0; i < sz; ++i )
      delete(streams_[i]);
  }

// private helpers
private:
  std::string usage() const {
    std::string msg = "Usage:\n";
    msg += "[-c][-n][-p] MasterFile ComparatorFile\n";
    msg += "  where:\n";
    msg += "    -c means 'include id of nearest ComparatorFile entry'\n";
    msg += "    -n means 'no exact overlaps allowed'\n";
    msg += "    -p means 'polarity sensitive'\n";
    return(msg);
  }

private:
  bool allowCompleteOverlaps_;
  bool strandCompare_;
  bool closestCoord_;
  std::size_t current_;
  std::vector<std::string> allFiles_;
  std::vector<std::ifstream*> streams_;
};


} // namespace FeatDist


#endif // INPUT_FEATDIST_H
