// Macro Guard
#ifndef BED_READER_FEAT_DIST_H
#define BED_READER_FEAT_DIST_H

// Files included
#include "Assertion.hpp"
#include "ByLine.hpp"
#include "FeatureDistanceDefns.hpp"


namespace FeatDist {

//===========
// BedReader
//===========
struct BedReader {
  // Constructor
  explicit BedReader(std::istream_iterator<ByLine>& is)
    : getNext_(true), coords_(std::make_pair(0, 0)),
      input_(is), eos_(), strand_(PLUS), id_(NOID), numCols_(0) {

    std::string msg = "Input has no data";
    Assert<InputError>(input_ != eos_, msg);
    while ( input_ != eos_ ) { /* find first row of data */
      if (
          input_->find("chr") == 0 ||
          input_->find("chR") == 0 ||
          input_->find("cHr") == 0 ||
          input_->find("cHR") == 0 ||
          input_->find("Chr") == 0 ||
          input_->find("ChR") == 0 ||
          input_->find("CHr") == 0 ||
          input_->find("CHR") == 0
         ) {
        readIn();
        ++input_;
        break;
      }
    } // while
    getNext_ = false;
  }

  // Public methods
  const std::string& Chrom() const { // call only after ReadLine()
    return(chrom_);
  }

  bool HasNext() {
    if ( !getNext_ )
      return(true);
    if ( input_ == eos_ || ReadLine() == NADA )
      return(false);
    getNext_ = false; // ReadLine() called
    return(true);
  }

  const std::string& ID() const {
    return(id_);
  }

  long NumberApplicableColumns() const {
    return(numCols_);
  }

  void PushBack() {
    getNext_ = false;
  }

  const PType& ReadLine() {
    static ByLine bl;
    if ( getNext_ ) {
      if ( input_->empty() )
        return(NADA);
      readIn();
      ++input_;
    }
    getNext_ = true;
    return(coords_);
  }

  StrandPolarity Strand() const {
    return(strand_);
  }

  virtual ~BedReader() { /* */ }

private: // helpers
  void readIn() {
    /* Assume input row is good --> no error checking here */
    std::string::size_type chrPos   = input_->find(SEP);
    std::string::size_type startPos = input_->find(SEP, ++chrPos);
    std::string::size_type endPos   = input_->find(SEP, ++startPos);
    static const std::string::size_type sz = CHROMOSOME.size();
    chrom_ = input_->substr(sz, chrPos-sz-1);
    static const int base = 10;
    // faster than coords_.first = convert<unsigned long>(...)
    coords_.first = strtoul(
                            input_->substr(chrPos, startPos-chrPos-1).c_str(),
                            NULL,
                            base
                           );

    numCols_ = 3;
    if ( endPos != std::string::npos ) {
      coords_.second = strtoul(
                             input_->substr(startPos, endPos-startPos).c_str(),
                             NULL,
                             base
                              );

      // Look for name, skip score, grab strand where present.
      ++numCols_;
      std::string::size_type namePos = input_->find(SEP, ++endPos);
      std::string::size_type scorePos, strandPos;
      if ( namePos != std::string::npos ) {
        id_ = input_->substr(endPos, namePos-endPos);
        ++numCols_;
        scorePos  = input_->find(SEP, ++namePos);
        if ( scorePos != std::string::npos ) {
          ++numCols_;
          strandPos = input_->find(SEP, ++scorePos);
          std::string tmp;
          if ( strandPos != std::string::npos )
            tmp = input_->substr(scorePos, 1);
          else
            tmp = input_->substr(scorePos);          
          strand_ = (tmp[0] == '+') ? PLUS : MINUS;
        }
      }
      else
        id_ = input_->substr(endPos);
    }
    else
      coords_.second = strtoul(input_->substr(startPos).c_str(), NULL, base);
  }

private:
  bool getNext_;
  PType coords_;
  std::istream_iterator<ByLine> input_, eos_;
  std::string chrom_;
  StrandPolarity strand_;
  std::string id_;
  long numCols_;    
};

} // namespace FeatDist


#endif // BED_READER_FEAT_DIST_H
