#!/bin/csh -ef
# Author:  sjn
# Date:    Mon Sep 17 14:52:31 PDT 2007
# Project: Compression

# Check number of inputs
if ( $#argv < 2 || $#argv > 3 ) then
  printf "\tUsage 1: [--keepID] <5-column-input BED> <outfile-name>\n"
  printf "\tUsage 2: --wig <4-column-input WIG> <outfile-name>\n"
  printf "\tNOTE: stdin NOT allowed (for now)\n"
  exit
endif

# Some locals
set infile = "-"
set outputfilename = $argv[$#argv]
@ keepID = 0
@ isWIG = 0

# Check the input args for sanity
if ( $#argv == 2 ) then
  set infile = $1  
else if ( $#argv == 3 ) then
  if ( "$1" == "--wig" ) then
    set infile = $2
    @ isWIG++
  else if ( "$1" == "--keepID" ) then
    @ keepID++
    set infile = $2
  else
    printf "Unknown option: $1\n"
    exit
  endif
endif

# Look to ensure input file actually exists
if ( ! -e $infile ) then
  printf "Cannot find input file: $infile\n"
  exit
endif

# Add jarch extension
if ( "$outputfilename:e" != "jarch" ) then
  set tmp = $outputfilename
  if ( "$outputfilename:e" == "bz2" ) then
    set tmp = "$outputfilename:r.rename"
  else if ( "$outputfilename:e" == "gz" ) then
    set tmp = "$outputfilename:r.rename"
  endif
  set outputfilename = "$tmp.jarch"
endif

rm -f $outputfilename


# Deal with possibility that file was previously compressed with another technique
set decomp = "cat"
if ( "$infile:e" == "gz" ) then
  set decomp = "gunzip -c"
else if ( "$infile:e" == "bz2" ) then
  set decomp = "bunzip2 -c"
endif


# If it's a wig file, then need to create an empty 4th column prior to jarch
# If it's a bed file and keepID is set, then cut out 4th column and keep
#   The possibility of stdin makes this a bit more challenging when $keepID > 0
#   --> need to write stdin to a temporary file, then do all of the work.

if ( $isWIG > 0 ) then
  $decomp $infile \
    | awk 'BEGIN {FS="\t"; OFS=FS} ; { print $1, $2, $3, "i", $4 }' - \
    | jarch $outputfilename:t:r - \
   >! $outputfilename
else if ( $keepID > 0 ) then
  $decomp $infile \
    | cut -f4 - \
    | bzip2 -c -9 - \
   >! "column4.$outputfilename:t:r.bz2"

  $decomp $infile \
    | jarch $outputfilename:t:r - \
   >! $outputfilename

  tar --append --remove-file --file $outputfilename "column4.$outputfilename:t:r.bz2"
else
  $decomp $infile \
    | jarch $outputfilename:t:r - \
   >! $outputfilename
endif

exit 0
