// Author
//  Shane Neph : University of Washington

// Files included
#include "Assertion.hpp"
#include "ByLine.hpp"
#include "Conversion.hpp"
#include "Exception.hpp"
#include "Input.hpp"
#include "SignalMapDefns.hpp"
#include "StandardFiles.hpp"

//========================================
// See README for details on this program
//========================================

namespace {
  const unsigned char UCBIGGEST = std::numeric_limits<unsigned char>::max();
}

namespace SignalMap {

  //==============================================
  // static const's defined from SignalMapDefns.h
  //==============================================
  const CoordType MAXVAL = std::numeric_limits<CoordType>::max();
  const std::string NOCHROM = std::string(5, UCBIGGEST);
  const PType NADA = std::make_pair(MAXVAL, MAXVAL);
  const DType NOGOOD = std::make_pair(1, 0); // invalid range on purpose
  const TType NOGO = std::make_pair(NOCHROM, NOGOOD);
  const char SEP = '\t';
  const std::string NANUMBER = "NAN";
  const std::string CHROMOSOME = "chr";
  const long NUMFILES = 2;

} // namespace SignalMap


namespace { // unnamed

  //===============
  // local globals
  //===============
  SignalMap::TType updateCoords;
  SignalMap::PType currentCoords[SignalMap::NUMFILES];
  SignalMap::Bed* bedFiles[SignalMap::NUMFILES];
  bool doneFile[SignalMap::NUMFILES];
  std::string chrom[SignalMap::NUMFILES];
  double measure[SignalMap::NUMFILES];


  //======================
  // Necessary Prototypes
  //======================
  void cleanup();
  void doCompareCalculation(double, SignalMap::MType);
  bool getNextFileCoords(int);
  SignalMap::TType nextCompareLine(double, SignalMap::MType);

} // unnamed namespace


//========
// main()
//========
int main(int argc, char** argv) {
  typedef SignalMap::InputError IE;
  typedef SignalMap::ProgramError PE;
  bool error = false;
  try {
    // Check inputs; initialize variables
    SignalMap::Input input(argc, argv);
    for ( int i = 0; i < SignalMap::NUMFILES; ++i ) {
      bedFiles[i] = input.NextBed();
      doneFile[i] = false;
      currentCoords[i] = SignalMap::NADA;
      chrom[i] = SignalMap::NOCHROM;
      measure[i] = 0;
    } // for

    // Check for actual coords in each input file
    for ( int i = 0; i < SignalMap::NUMFILES; ++i )
      Assert<IE>(getNextFileCoords(i) != false, "Empty input file");
    doCompareCalculation(input.Percentage(), input.GetMType());
  } catch(IE& ie) {
    std::cerr << "Input Error Detected" << std::endl;
    std::cerr << ie.GetMessage() << std::endl;
    std::cerr << "  Usage: " << SignalMap::Input::Usage() << std::endl;
    error = true;
  } catch(PE& pe) {
    std::cerr << "Program Error Detected" << std::endl;
    std::cerr << pe.GetMessage() << std::endl;
    error = true;
  } catch(std::exception& err) {
    std::cerr << "Error: " << err.what() << std::endl;
    error = true;
  } catch(...) {
    std::cerr << "Unknown Error" << std::endl;
    error = true;
  }
  cleanup();
  return(error);
}



// Function implementations
namespace { // unnamed

  // Lazy (using) declarations and typedefs
  typedef SignalMap::Bed::innards Innards;

  using SignalMap::CoordType;
  using SignalMap::MType;
  using SignalMap::PType;
  using SignalMap::TType;
  using SignalMap::NADA;
  using SignalMap::NANUMBER;
  using SignalMap::NOGO;
  using SignalMap::NOGOOD;
  using SignalMap::MAX;
  using SignalMap::MEAN;
  using SignalMap::MEDIAN;
  using SignalMap::MIN;
  using SignalMap::SUM;
  using SignalMap::VARIANCE;

  //===========
  // cleanup()
  //===========
  void cleanup() {
    for ( int i = 0; i < SignalMap::NUMFILES; ++i ) {
      if ( bedFiles[i] )
        delete(bedFiles[i]);
    } // for
  }

  //==========
  // record()
  //==========
  void record(const TType& t) {
    std::cout << t.second.first
              << SignalMap::SEP
              << t.second.second
              << std::endl;
  }

  //==========
  // record()
  //==========
  void record(const std::string& str, const SignalMap::PType& p) {
    std::cout << str
              << SignalMap::SEP
              << p.first
              << std::endl;
  }

  //========================
  // doCompareCalculation()
  //========================
  void doCompareCalculation(double specified, MType mtype) {
    bool done = false;
    const int ref = 0; // 0 is master index
    const PType zeroes = std::make_pair(0, 0);
    while ( !done ) {
      updateCoords = nextCompareLine(specified, mtype);
      if ( updateCoords == NOGO )
        break;                
      else if ( updateCoords.second == NOGOOD )
        record(NANUMBER, zeroes);
      else
        record(updateCoords);
      getNextFileCoords(ref);
    } // while     
  }

  //=============
  // calculate()
  //=============
  double calculate(const std::vector<double>& values, MType mtype) {
    /* This routine won't be called if values.empty() is true */
    static const double zero = 0;
    double size = static_cast<double>(values.size());
    if ( mtype == MEAN )
      return(std::accumulate(values.begin(), values.end(), zero) / size);
    else if ( mtype == MAX )
      return(*std::max_element(values.begin(), values.end()));
    else if ( mtype == MIN )
      return(*std::min_element(values.begin(), values.end()));
    else if ( mtype == SUM )
      return(std::accumulate(values.begin(), values.end(), zero));
    else if ( mtype == MEDIAN ) {
      std::vector<double> cpy(values);
      std::vector<double>::iterator rIter = cpy.begin();
      std::size_t ssz = cpy.size();
      if ( ssz % 2 != 0 ) {
        rIter += (ssz-1)/2;
        std::nth_element(cpy.begin(), rIter, cpy.end());
        return(*rIter);
      }
      rIter += ssz/2-1;
      std::nth_element(cpy.begin(), rIter, cpy.end());
      double one = *rIter;
      std::nth_element(cpy.begin(), ++rIter, cpy.end());
      return((one + *rIter)/2);
    }
    else { // mtype == VARIANCE
      double mean = calculate(values, MEAN);
      double sum = 0;
      std::size_t sz = values.size();
      if ( sz == 0 )
        return(0);
      for ( std::size_t i = 0; i < sz; ++i ) {
        double val = values[i] - mean;
        sum += val * val;
      } // for
      if ( sz == 1 )
        ++sz;
      return(sum / (sz - 1));
    }
  }

  //=====================
  // getNextFileCoords()
  //=====================
  bool getNextFileCoords(int i) {
    // Merge coordinates within a file
    if ( !bedFiles[i]->HasNext() ) {
      doneFile[i] = true;
      return(false);
    }
    currentCoords[i] = bedFiles[i]->ReadLine();
    chrom[i]         = bedFiles[i]->Chrom();
    measure[i]       = bedFiles[i]->Measure();
    return(true);
  }

  //===================
  // intersectOverlap()
  //===================
  PType intersectOverlap(const PType& p1, const PType& p2) {
    CoordType min = std::max(p1.first, p2.first);
    CoordType max = std::min(p1.second, p2.second);
    return((max >= min) ? std::make_pair(min, max) : NADA);
  }

  //================
  // mergeOverlap()
  //================
  PType mergeOverlap(const PType& p1, const PType& p2) {
    PType toRtn = NADA;
    if ( p1.first < p2.first ) {
      if ( p1.second >= p2.first - 1 ) { // p2.first >= 1
        toRtn.first = p1.first;
        toRtn.second = std::max(p1.second, p2.second);
      }
    }
    else if ( p1.first > p2.first ) {
      if ( p2.second >= p1.first - 1 ) { // p1.first >= 1
        toRtn.first = p2.first;
        toRtn.second = std::max(p1.second, p2.second);
      }
    }
    else { // p1.first == p2.first
      toRtn.first = p1.first;
      toRtn.second = std::max(p1.second, p2.second);
    }
    return(toRtn);
  }

  //===============
  // sizeOverlap()
  //===============
  CoordType sizeOverlap(const std::vector<PType>& p) {
    if ( p.empty() )
      return(0);

    std::vector<PType>::const_iterator i = p.begin(), j = p.end();
    PType tmp = *i, keep = *i;
    CoordType toRtn = 0;
    while ( ++i != j ) {
      tmp = mergeOverlap(keep, *i);
      if ( tmp == NADA ) {
        toRtn += (keep.second - keep.first + 1);
        keep = *i;
      }
      else
        keep = tmp;
    } // while
    toRtn += (keep.second - keep.first + 1);
    return(toRtn);
  }

  //===================
  // nextCompareLine()
  //===================
  TType nextCompareLine(double specified, MType mtype) {
    static const int ref = 0;
    static const int noRef = 1;
    bool done = false;
    bool misaligned = false, refBigger = false;
    TType toRtn = NOGO;
    if ( doneFile[ref] )
      return(NOGO);
    std::vector<double> nextMeasure;
    std::vector<PType> overlapCoords;
    CoordType size = 0;
    PType overlap;
    double total, percent;
    std::list<Innards> toReplay;

    while ( !done ) { // keep looking
      if ( doneFile[noRef] ) {
        if ( nextMeasure.empty() )
          toRtn.second = NOGOOD;
        else
          toRtn.second = std::make_pair(calculate(nextMeasure, mtype),
                                        sizeOverlap(overlapCoords));
        break;
      }

      // See if ref && nonRef are misaligned | nonoverlapping
      misaligned = (chrom[ref] != chrom[noRef]);
      if ( misaligned ) {
        refBigger = (chrom[ref] > chrom[noRef]);
        overlap = NADA;
      }
      else {
        overlap = intersectOverlap(currentCoords[ref], currentCoords[noRef]);
        misaligned = false;
      }
      
      if ( misaligned || overlap == NADA ) {
        if (  /* noRef is ahead of ref */
            (misaligned && !refBigger) ||
            (!misaligned && currentCoords[noRef].first > currentCoords[ref].second)
           ) {
          toReplay.push_front(bedFiles[noRef]->GetState());
          if ( nextMeasure.empty() )
            toRtn.second = NOGOOD;
          else
            toRtn.second = std::make_pair(calculate(nextMeasure, mtype),
                                          sizeOverlap(overlapCoords));
          break;
        }
        else // ref ahead of noRef; catch up
          getNextFileCoords(noRef);
      }
      else { // overlap exists
        size = overlap.second - overlap.first + 1;
        total = currentCoords[noRef].second - currentCoords[noRef].first + 1;
        percent = (static_cast<double>(size) / total) * 100;
        if ( percent >= specified ) {
          nextMeasure.push_back(measure[noRef]);
          overlapCoords.push_back(overlap);
        }

        toReplay.push_front(bedFiles[noRef]->GetState());
        getNextFileCoords(noRef);
      }
    } // while

    // May need to reuse noRef's values on next ref iteration
    std::list<Innards>::iterator i = toReplay.begin(), j = toReplay.end();
    bool restore = (i != j);
    while ( i != j )
      bedFiles[noRef]->Push(*i++);

    if ( restore ) {
      doneFile[noRef] = false;
      getNextFileCoords(noRef);
    }
    toRtn.first = chrom[ref];
    return(toRtn);
  }

} // unnamed
