// Author
//  Shane Neph : University of Washington

// Macro Guard
#ifndef SIGNALMAP_INPUT_H
#define SIGNALMAP_INPUT_H

// Files included
#include "Assertion.hpp"
#include "BedFile.hpp"
#include "Exception.hpp"
#include "SignalMapDefns.hpp"
#include "StandardFiles.hpp"

namespace SignalMap {

  //=======
  // Input
  //=======
  struct Input {

    // Constructor
    Input(int argc, char** argv) {
      static const double DEFAULTPERCENTAGE = 10;

      Assert<InputError>(argc > 3, "Not enough inputs");
      mType_ = getMType(argv[1]);
      if ( argc == 4 ) {
        percent_ = DEFAULTPERCENTAGE;
        allFiles_ = std::make_pair(argv[2], argv[3]);
      }
      else if ( argv[2] == std::string("-%") ) {
        std::string tmp(argv[3]), num("0123456789.");
        std::string::size_type pos = tmp.find_first_not_of(num);
        Assert<InputError>(pos == std::string::npos, "Bad % value");
        percent_= atof(argv[3]);
        Assert<InputError>(argc > 5, "Not enough input files");
        allFiles_ = std::make_pair(argv[4], argv[5]);
      }

      current_ = 0;
      if ( allFiles_.first != "-" ) {
        std::ifstream check1(allFiles_.first.c_str());
        Assert<InputError>(check1, "Cannot find file " + allFiles_.first);
      }

      if ( allFiles_.second != "-" ) {
        std::ifstream check2(allFiles_.second.c_str());
        Assert<InputError>(check2, "Cannot find file " + allFiles_.second);
      }
      Assert<InputError>(allFiles_.first != "-" || allFiles_.second != "-",
                         "Cannot use more than 1 '-' input");
    }

    // Public methods
    MType GetMType() const {
      return(mType_);
    }
    double Percentage() const {
      return(percent_);
    }
    Bed* NextBed() {
      bool allFields = (current_++ != 0);
      if ( !allFields ) // .bed for MasterFile
        return(new SignalMap::Bed(allFiles_.first, allFields));
      else
        return(new SignalMap::Bed(allFiles_.second, allFields));
    }
    static std::string Usage() {
      std::string use = "-max|mean|median|min|sum|var";
      use += " [-% overlap] MasterFile File2";
      return(use);
    }

  // private helpers
  private:
    MType getMType(const std::string& type) {
      std::string str = getLower(type);
      if ( str == "-max" )
        return(MAX);
      else if ( str == "-min" )
        return(MIN);
      else if ( str == "-mean" )
        return(MEAN);
      else if ( str == "-median" )
        return(MEDIAN);
      else if ( str == "-sum" )
        return(SUM);
      else if ( str == "-var" )
        return(VARIANCE);
      throw(InputError("Unknown calculation type: " + str));
    }

    std::string getLower(const std::string& str) {
      std::string toRtn;
      std::string::size_type sz = str.size();
      for ( std::size_t i = 0; i < sz; ++i )
        toRtn += std::tolower(str[i]);
      return(toRtn);
    }

  private:
    MType mType_;
    int current_;
    double percent_;
    std::pair<std::string, std::string> allFiles_;
  };


} // namespace SignalMap

#endif // SIGNALMAP_INPUT_H
