// Author
//  Shane Neph : University of Washington

// Macro Guard
#ifndef BED_FILE_SIGNALMAP_H
#define BED_FILE_SIGNALMAP_H

// Files included
#include "SignalMapDefns.hpp"
#include "StandardFiles.hpp"


namespace SignalMap {

  //=====
  // Bed
  //=====
  struct Bed {

    //======================
    // innards Nested Class
    //======================
    struct innards {
        innards() : coords_(std::make_pair(0, 0)), chrom_(""), measure_(0),
                    allFields_(true) { /* */ }

      PType coords_;
      std::string chrom_;
      double measure_;
      bool allFields_;
    };


    //=====================
    // Constructor for BED
    //=====================
    Bed(const std::string& s, bool allFields) : 
                  useCin_(s=="-"),
                  inFile_(useCin_ ? 0 : new std::ifstream(s.c_str())),
                  input_(useCin_ ? std::cin : *inFile_) {

      std::istream_iterator<ByLine> eof;
      while ( input_ != eof ) {
        ByLine bl(*input_++);
        if (
            bl.find("chr") == 0 || bl.find("chR") == 0 ||
            bl.find("cHr") == 0 || bl.find("cHR") == 0 ||
            bl.find("Chr") == 0 || bl.find("ChR") == 0 ||
            bl.find("CHr") == 0 || bl.find("CHR") == 0
           ) {
          readIn(bl, allFields);
          Push(inn_);
          return;
        }
      } // while
      std::string msg = "Bad input detected ; could not find first line";
      Assert<InputError>(false, msg);
    }

    //================
    // Public methods
    //================
    std::string Chrom() const { // call only after ReadLine()
      return(inn_.chrom_);
    }

    bool HasNext() {
      static std::istream_iterator<ByLine> eof;
      if ( !history_.empty() )
        return(true);
      if ( input_ == eof )
        return(false);

      innards tmp = inn_;
      if ( ReadLine() == NADA ) {
        inn_ = tmp;
        return(false);
      }
      Push(inn_);
      inn_ = tmp;
      return(true);
    }

    const innards& GetState() const {
      return(inn_);
    }

    double Measure() const {
      return(inn_.measure_);
    }

    void Push(const Bed::innards& i) {
      history_.push_front(i);
    }

    const PType& ReadLine() {
      static ByLine bl;
      if ( history_.empty() ) {
        bl = *input_++;
        if ( bl.empty() )
          return(NADA);
        readIn(bl, inn_.allFields_);
      }
      else {
        readIn(*(history_.begin()));
        history_.erase(history_.begin());
      }
      return(inn_.coords_);
    }

    //====================
    // End Public Methods
    //====================

    ~Bed() {
      if ( inFile_ )
        delete(inFile_);
    }

  private: // helpers
    Bed(const Bed&); // input_ prevents copy construction

    void readIn(const std::string& bl, bool includeMeasure) {
      read(bl, includeMeasure);
    }

    void readIn(const innards& i) {
      inn_ = i;
    }

    void read(const std::string& bl, bool includeMeasure) {
      /* Assume input row is good --> no error checking here */
      std::string::size_type pos = bl.find(SEP);
      std::string::size_type nex = bl.find(SEP, ++pos);
      std::string::size_type las = bl.find(SEP, ++nex);
      static const std::string::size_type sz = CHROMOSOME.size();
      inn_.chrom_ = bl.substr(sz, pos-sz-1);
      static const int bs = 10;
      inn_.coords_.first = strtoul(bl.substr(pos, nex-pos).c_str(), NULL, bs);
      inn_.coords_.second = strtoul(bl.substr(nex, las-nex).c_str(), NULL, bs);
      inn_.coords_.second -= 1; // [a,b) --> [a,b-1]
      inn_.measure_ = 0;
      inn_.allFields_ = includeMeasure;
      if ( !includeMeasure )
        return;
      pos = bl.find(SEP, ++las);
      las = bl.find(SEP, ++pos);
      if ( las == std::string::npos ) {
        inn_.measure_ = atof(bl.substr(pos).c_str());
        return;
      }
      inn_.measure_ = atof(bl.substr(pos, las-pos).c_str());
    }

  private:
    bool useCin_;
    std::ifstream* inFile_;
    std::istream_iterator<ByLine> input_;
    std::list<innards> history_;
    innards inn_;
  };

} // namespace SignalMap

#endif // BED_FILE_SIGNALMAP_H
