// Author
//  Shane Neph : University of Washington

// Files included
#include "Assertion.hpp"
#include "BedReader.hpp"
#include "Input.hpp"
#include "SetOpsDefns.hpp"
#include "StandardFiles.hpp"


//=========================================================
// See README for details on how this program can be used.
//=========================================================


namespace { // unnamed

//===============
// local globals
//===============
SetOperations::TType updateCoords, nextCoords;
std::vector<SetOperations::PType> currentCoords;
std::vector<SetOperations::BedReader*> bedFiles;
std::vector<bool> doneFile;
std::vector<std::string> chrom;
std::vector<std::string const*> rest;

const unsigned char UCBIGGEST = std::numeric_limits<unsigned char>::max();

} // unnamed




namespace SetOperations {


//===================================
// Define externs from SetOpsDefns.h
//===================================
const PType NADA = std::make_pair(UCBIGGEST, UCBIGGEST-1); // invalid range on purpose
const std::string MINSTRING = std::string(20, 0);
const std::string NOCHROM = std::string(5, UCBIGGEST);
const std::string NOCHROMMARKER = std::string(4, UCBIGGEST);
const TType NOGO = std::make_pair(NOCHROM, NADA);
const TType CALLAGAIN = std::make_pair(NOCHROMMARKER, NADA);
const char SEP = '\t';
const std::string CHROMOSOME = "chr";
const std::string NOREST = "";
const std::string USESTDIN = "use standard input";


//============
// Prototypes
//============
void  cleanup();
void  doComplement(int);
void  doDifference(int);
void  doElementOf(int, double, bool, bool);
void  doIntersection(int);
void  doMerge(int);
void  doSymmetricDifference(int);
void  doUnionAll(int);
PType intersectOverlap(const PType&, const PType&);
bool  getNextFileCoords(int);
bool  getNextFileLine(int i);
TType getNextMerge(std::queue<TType>&, int, int);
PType mergeOverlap(const PType&, const PType&);
TType nextComplementLine(int);
TType nextDifferenceLine(int, TType&);
TType nextElementOfLine(int, std::queue<TType>&, double, bool, bool);
TType nextIntersectLine(int);
TType nextMergeLine(int, int);
TType nextSymmetricDiffLine(int);
TType nextUnionAllLine(int, int&);
void  record(const PType&, const std::string&);


} // namespace SetOperations


//========
// main()
//========
int main(int argc, char** argv) {
  using namespace SetOperations;

  bool error = false;
  try {
    // Check inputs; initialize variables
    Input input(argc, argv);
    ModeType modeType = input.GetModeType();
    int numFiles = input.NumberFiles();
    for ( int i = 0; i < numFiles; ++i ) {
      bedFiles.push_back(input.NextBed());
      doneFile.push_back(false);
      currentCoords.push_back(NADA);
      chrom.push_back(NOCHROM);
      rest.push_back(&NOREST);
    } // for

    // Check for actual coords in each input file
    if ( modeType == UNIONALL ) { // no merging
      for ( int i = 0; i < numFiles; ++i )
        getNextFileLine(i);
    }
    else {
      int start = 0;
      if ( modeType == ELEMENTOF || modeType == NOTELEMENTOF ) {
        /* no merging of reference file */
        getNextFileLine(start);
        ++start;
      }

      for ( int i = start; i < numFiles; ++i ) // safe to merge
        getNextFileCoords(i);
    }

    // Iterate through all input files and output results
    bool doInvert = true, noInvert = false;
    switch ( modeType ) {
      case COMPLEMENT:
        doComplement(numFiles);
        break;
      case DIFFERENCE:
        doDifference(numFiles);
        break;
      case ELEMENTOF:
        doElementOf(numFiles, input.Threshold(),
                    input.UsePercentage(), noInvert);
        break;
      case INTERSECTION:
        doIntersection(numFiles);
        break;
      case MERGE:
        doMerge(numFiles);
        break;
      case NOTELEMENTOF:
        doElementOf(numFiles, input.Threshold(),
                    input.UsePercentage(), doInvert);
        break;
      case SYMMETRIC_DIFFERENCE:
        doSymmetricDifference(numFiles);
        break;
      case UNIONALL:
        doUnionAll(numFiles);
        break;
      default:
        Assert<InputError>(false, "Unsupported mode");
    };
  } catch(HelpException& he) {
    std::cerr << he.GetMessage() << std::endl;
    error = true;
  } catch(InputError& ie) {
    std::cerr << "Input Error Detected" << std::endl;
    std::cerr << ie.GetMessage() << std::endl;
    error = true;
  } catch(ProgramError& pe) {
    std::cerr << "Program Error Detected" << std::endl;
    std::cerr << pe.GetMessage() << std::endl;
    error = true;
  } catch(std::exception& err) {
    std::cerr << "Error: " << err.what() << std::endl;
    error = true;
  } catch(...) {
    std::cerr << "Unknown Error - is input in .bed format?" << std::endl;
    error = true;
  }
  cleanup();
  return(error);
}



// Function implementations
namespace SetOperations {

//===========
// cleanup()
//===========
void cleanup() {
  std::vector<BedReader*>::iterator i = bedFiles.begin(), j = bedFiles.end();
  while ( i != j ) {
    if ( *i )
      delete(*i);
    ++i;
  } // while
}

//==========
// record()
//==========
void record(const TType& p, const std::string& rest = "") {
  std::cout << CHROMOSOME << p.first
            << SEP << p.second.first
            << SEP << (p.second.second + 1); // [a, b-1] --> [a, b)
  if ( !rest.empty() )
    std::cout << SEP << rest;
  std::cout << std::endl;
}

//================
// doComplement()
//================
void doComplement(int numFiles) {
  bool done = false;
  while ( !done ) {
    updateCoords = nextComplementLine(numFiles);
    if ( updateCoords == NOGO )
      break;
    else if ( updateCoords != CALLAGAIN )
      record(updateCoords);
  } // while
}

//================
// doDifference()
//================
void doDifference(int numFiles) {
  bool done = false;
  const int noRef = 1; // 0 is master index
  TType nextMerge = nextMergeLine(noRef, numFiles);
  while ( !done ) {
    updateCoords = nextDifferenceLine(numFiles, nextMerge);
    if ( updateCoords == NOGO )
      break;
    else if ( updateCoords != CALLAGAIN )
      record(updateCoords);
  } // while
}

//===============
// doElementOf()
//===============
void doElementOf(int numFiles, double thres, bool usePerc, bool invert) {
  bool done = false;
  const int ref = 0;
  const int noRef = 1;
  std::queue<TType> q;
  q.push(nextMergeLine(noRef, numFiles));
  while ( !done ) {
    updateCoords = nextElementOfLine(numFiles, q, thres, usePerc, invert);
    if ( updateCoords == NOGO )
      break;
    else if ( updateCoords != CALLAGAIN )
      record(updateCoords, *rest[ref]);
    getNextFileLine(ref); // unmerged
  } // while
}

//==================
// doIntersection()
//==================
void doIntersection(int numFiles) {
  bool done = false;
  while ( !done ) {
    updateCoords = nextIntersectLine(numFiles);
    if ( updateCoords == NOGO )
      break;
    record(updateCoords);
  } // while
}

//===========
// doMerge()
//===========
void doMerge(int numFiles) {
  bool done = false;
  while ( !done ) {
    updateCoords = nextMergeLine(0, numFiles);
    if ( updateCoords == NOGO )
      break;
    record(updateCoords);
  } // while
}

//=========================
// doSymmetricDifference()
//=========================
void doSymmetricDifference(int numFiles) {
  /* must cache last result to ensure all overlaps are caught */
  bool done = false, first = true;
  TType nextDiff = NOGO, toRecord = NOGO;
  PType overlap = NADA;
  while ( !done ) {
    nextDiff = nextSymmetricDiffLine(numFiles);
    if ( nextDiff == NOGO ) {
      if ( !first )
        record(toRecord);
      break;
    }
    else if ( nextDiff == CALLAGAIN )
      nextDiff = NOGO;
    else {
      if ( toRecord.first != nextDiff.first ) {
        if ( !first )
          record(toRecord);
        toRecord = nextDiff;
      }
      else {
        overlap = mergeOverlap(nextDiff.second, toRecord.second);
        if ( overlap == NADA ) {
          if ( !first )
            record(toRecord);
          toRecord = nextDiff;
        }
        else
          toRecord.second = overlap;
      }
      first = false;
    }
  } // while
}

//==============
// doUnionAll()
//==============
void doUnionAll(int numFiles) {
  /* If inputs have duplicate entries, output will too */
  bool done = false;
  int marker = -1, val = -1;
  while ( !done ) {
    updateCoords = nextUnionAllLine(numFiles, marker);
    if ( updateCoords == NOGO )
      break;
    record(updateCoords, *rest[marker]);
    getNextFileLine(marker); // unmerged
    marker = val;
  } // while
}

//=====================
// getNextFileCoords()
//=====================
bool getNextFileCoords(int i) {
  // Merge coordinates within a file
  if ( !bedFiles[i]->HasNext() ) {
    doneFile[i] = true;
    return(false);
  }
  static PType update, next;
  currentCoords[i] = bedFiles[i]->ReadLine();
  chrom[i] = bedFiles[i]->Chrom();
  if ( !bedFiles[i]->HasNext() )
    return(true); // doneFile[i] set to true on next call

  while ( !doneFile[i] ) {
    next = bedFiles[i]->ReadLine();
    if ( chrom[i] == bedFiles[i]->Chrom() ) {
      update = mergeOverlap(next, currentCoords[i]);
      if ( update != NADA )
        currentCoords[i] = update;
      else {
        bedFiles[i]->PushBack();
        break;
      }
    }
    else {
      bedFiles[i]->PushBack();
      break;
    }

    if ( !bedFiles[i]->HasNext() )
      break;
  } // while
  return(!doneFile[i]);
}

//===================
// getNextFileLine()
//===================
bool getNextFileLine(int i) {
  // Read next line from index i without merging
  if ( !bedFiles[i]->HasNext() ) {
    doneFile[i] = true;
    return(false);
  }
  currentCoords[i] = bedFiles[i]->ReadLine();
  chrom[i] = bedFiles[i]->Chrom();
  rest[i] = bedFiles[i]->Rest();
  return(true); // doneFile[i] checked on next call
}

//================
// getNextMerge()
//================
TType getNextMerge(std::queue<TType>& mergeList, int start, int numFiles) {
  if ( mergeList.empty() )
    return(nextMergeLine(start, numFiles));
  TType toRtn = mergeList.front();
  mergeList.pop();
  return(toRtn);
}

//===================
// intersectOverlap()
//===================
PType intersectOverlap(const PType& p1, const PType& p2) {
  unsigned long min = std::max(p1.first, p2.first);
  unsigned long max = std::min(p1.second, p2.second);
  return((max >= min) ? std::make_pair(min, max) : NADA);
}

//================
// mergeOverlap()
//================
PType mergeOverlap(const PType& p1, const PType& p2) {
  PType toRtn = NADA;
  if ( p1.first < p2.first ) {
    if ( p1.second >= p2.first - 1 ) { // p2.first >= 1
      toRtn.first = p1.first;
      toRtn.second = std::max(p1.second, p2.second);
    }
  }
  else if ( p1.first > p2.first ) {
    if ( p2.second >= p1.first - 1 ) { // p1.first >= 1
      toRtn.first = p2.first;
      toRtn.second = std::max(p1.second, p2.second);
    }
  }
  else { // p1.first == p2.first
    toRtn.first = p1.first;
    toRtn.second = std::max(p1.second, p2.second);
  }
  return(toRtn);
}

//======================
// nextComplementLine()
//======================
TType nextComplementLine(int numFiles) {
  static TType last = NOGO;
  if ( last == NOGO ) {
    last = nextMergeLine(0, numFiles);
    if ( last == NOGO )
      return(NOGO);
  }
  nextCoords = nextMergeLine(0, numFiles);
  if ( nextCoords == NOGO )
    return(NOGO);
  TType toRtn = std::make_pair(last.first,
                               std::make_pair(last.second.second+1,
                                              nextCoords.second.first-1));
  if ( nextCoords.first != last.first ) {
    last = nextCoords;
    return(CALLAGAIN);
    // direct recursion removed to prevent any possible stack overflow
    // return(nextComplementLine(numFiles)); // curse some more
  }
  last = nextCoords;
  return(toRtn);
}

//======================
// nextDifferenceLine()
//======================
TType nextDifferenceLine(int numFiles, TType& nextMerge) {
  // Index 0 is the reference file
  static const int ref = 0;
  static const int noRef = 1;
  TType toRtn = NOGO;

  if ( doneFile[ref] )
    return(NOGO);
  else if ( nextMerge == NOGO ) { // everything else is a difference
    toRtn = std::make_pair(chrom[ref], currentCoords[ref]);
    getNextFileCoords(ref); // doneFile[ref] checked on next call
    return(toRtn);
  }

  // Increment nextMerge until its back within range of ref
  while ( nextMerge.first < chrom[ref] ||
          (nextMerge.first == chrom[ref] &&
           nextMerge.second.second < currentCoords[ref].first)
        ) {
    nextMerge = nextMergeLine(noRef, numFiles);
    if ( nextMerge == NOGO ) { // always true after first true
      toRtn = std::make_pair(chrom[ref], currentCoords[ref]);
      getNextFileCoords(ref); // doneFile[ref] checked on next call
      return(toRtn);
    }
  } // while

  // Compare orientation of nextMerge and currentCoords[ref]
  if ( nextMerge.first > chrom[ref] || 
    nextMerge.second.first > currentCoords[ref].second ) {
    /* no reference overlap */
    toRtn = std::make_pair(chrom[ref], currentCoords[ref]);
    getNextFileCoords(ref); // doneFile[ref] checked on next call
  }
  else if ( nextMerge.second.first <= currentCoords[ref].first &&
            nextMerge.second.second >= currentCoords[ref].second ) {
    /* complete reference overlap */
    getNextFileCoords(ref); // doneFile[ref] checked on next call
    toRtn = CALLAGAIN;
    // direct recursion removed to prevent any possible stack overflow
    // return(nextDifferenceLine(numFiles, nextMerge)); // curse some more
  }
  else if ( nextMerge.second.first > currentCoords[ref].first ) {
    /* difference found up to nextMerge.second.first - 1 */
    toRtn.first = chrom[ref];
    toRtn.second = std::make_pair(currentCoords[ref].first,
                                  nextMerge.second.first-1);
    currentCoords[ref].first = nextMerge.second.second + 1;
    while ( nextMerge.first == chrom[ref] &&
            nextMerge.second.second >= currentCoords[ref].second ) {
      if ( !getNextFileCoords(ref) )
        break;
    } // while
  }
  else { // nextMerge.second.first <= currentCoords[ref].first
    /* need to shorten current currentCoords[ref]  */
    currentCoords[ref].first = nextMerge.second.second + 1;
    toRtn = CALLAGAIN;
    // direct recursion removed to prevent any possible stack overflow
    // return(nextDifferenceLine(numFiles, nextMerge)); // curse some more
  }
  return(toRtn);
}

//=====================
// nextElementOfLine()
//=====================
TType nextElementOfLine(int numFiles, std::queue<TType>& mergeList,
                        double threshold, bool usePercent, bool invert) {
  // Index 0 is the reference file
  static const int ref = 0;
  static const int noRef = 1;
  TType toRtn = NOGO;

  if ( doneFile[ref] )
    return(NOGO);
  TType nextMerge = getNextMerge(mergeList, noRef, numFiles);
  if ( nextMerge == NOGO ) {
    if ( !invert ) // ref cannot be an element of nada
      return(NOGO);
    return(std::make_pair(chrom[ref], currentCoords[ref]));
  }

  // Increment nextMerge until its back within range of ref
  while ( nextMerge.first < chrom[ref] || 
         (nextMerge.first == chrom[ref] &&
         nextMerge.second.second < currentCoords[ref].first)
        ) {
    nextMerge = getNextMerge(mergeList, noRef, numFiles);
    if ( nextMerge == NOGO ) {
      if ( !invert ) // ref cannot be an element of nada
        return(NOGO);
      return(std::make_pair(chrom[ref], currentCoords[ref]));
    }
  } // while

  bool done = false;
  double rangeOverlap = 0;
  double range = currentCoords[ref].second - currentCoords[ref].first + 1;
  toRtn.first = chrom[ref];
  toRtn.second = currentCoords[ref];
  std::vector<TType> toPush;
  toPush.push_back(nextMerge);
  while ( !done ) {
    // Compare orientation of nextMerge and currentCoords[ref]
    if ( nextMerge.first > chrom[ref] || 
         nextMerge.second.first > currentCoords[ref].second ) {
      /* no reference overlap */
      break;
    }
    else { // partial or complete reference overlap
      PType lap = intersectOverlap(nextMerge.second, currentCoords[ref]);
      rangeOverlap += lap.second - lap.first + 1;
      nextMerge = getNextMerge(mergeList, noRef, numFiles);
      if ( nextMerge == NOGO )
        break;
      toPush.push_back(nextMerge);
    }
  } // while

  if ( mergeList.empty() ) {
    for ( std::size_t i = 0; i < toPush.size(); ++i )
      mergeList.push(toPush[i]);
  }
  else { /* must push previously popped items to front of queue again */
    std::vector<TType> tmp;
    while ( !mergeList.empty() ) {
      tmp.push_back(mergeList.front());
      mergeList.pop();
    } // while
    for ( std::size_t i = 0; i < toPush.size(); ++i )
      mergeList.push(toPush[i]);
    for ( std::size_t i = 0; i < tmp.size(); ++i )
      mergeList.push(tmp[i]);
  }
  bool isElement = (rangeOverlap / range >= threshold);
  if ( !usePercent ) // measure in bps of overlap
    isElement = (rangeOverlap >= threshold);
  if ( invert ) // invert logic --> not element of
    return(isElement ? CALLAGAIN : toRtn);
  return(isElement ? toRtn : CALLAGAIN);
}

//=====================
// nextIntersectLine()
//=====================
TType nextIntersectLine(int numFiles) {
  // Intersect coordinates between all  files.
  static const unsigned long k = std::numeric_limits<unsigned long>::max();
  static const PType init = std::make_pair(0, k);
  TType toRtn = std::make_pair(MINSTRING, init);
  for ( int i = 0; i < numFiles; ++i ) { // find maximum minimum
    if ( doneFile[i] ) // no more intersections
      return(NOGO);
    if ( chrom[i] >= toRtn.first ) {
      if ( 
        currentCoords[i].first > toRtn.second.first ||
        chrom[i] > toRtn.first
         )
        toRtn.second = currentCoords[i]; // starting point
      toRtn.first = chrom[i];
    }
  } // for

  int marker = -1; // only require 1 marker, even on ties
  unsigned long maxVal = std::numeric_limits<unsigned long>::max();
  for ( int i = 0; i < numFiles; ++i ) { // find next intersection
    while ( chrom[i] < toRtn.first ||
            (chrom[i] == toRtn.first &&
             currentCoords[i].second < toRtn.second.first) ) {
      if ( !getNextFileCoords(i) ) // no more intersections
        return(NOGO);
    } // while

    if ( chrom[i] > toRtn.first || 
         currentCoords[i].first > toRtn.second.second ) {

      toRtn = std::make_pair(chrom[i], currentCoords[i]);
      i = -1;
      marker = -1;
      maxVal = std::numeric_limits<unsigned long>::max();
      continue;
    }
    toRtn.second = intersectOverlap(toRtn.second, currentCoords[i]);
    if ( currentCoords[i].second < maxVal ) {
      maxVal = currentCoords[i].second;
      marker = i;
    }
  } // for
  getNextFileCoords(marker); // at least one file increment
  return(toRtn);
}

//=================
// nextMergeLine()
//=================
TType nextMergeLine(int start, int numFiles) {
  // Merge coordinates between files.
  static const PType inner = std::make_pair(
                               std::numeric_limits<unsigned long>::max(),
                               std::numeric_limits<unsigned long>::min()
                                           );
  TType toRtn = std::make_pair(NOCHROM, inner);
  bool anyNew = false;
  for ( int i = start; i < numFiles; ++i ) { // find minimum
    if ( !doneFile[i] ) {
      anyNew = true;
      if ( chrom[i] < toRtn.first ) {
        toRtn.first = chrom[i];
        toRtn.second = currentCoords[i];
      }
      else if ( chrom[i] == toRtn.first &&
                currentCoords[i].first < toRtn.second.first )
        toRtn.second = currentCoords[i];
    }
  } // for
  if ( !anyNew )
    return(NOGO);

  // Increment bedfiles in search of contiguous pieces
  for ( int i = start; i < numFiles; ++i ) {
    if ( doneFile[i] )
      continue;
    while ( chrom[i] == toRtn.first && 
            currentCoords[i].second <= toRtn.second.second ) {
      if ( !getNextFileCoords(i) ) // doneFile[i] is now true
        break;
    } // while

    // New max?
    if ( !doneFile[i] &&
         chrom[i] == toRtn.first &&
         currentCoords[i].first <= toRtn.second.second + 1 &&
         currentCoords[i].second > toRtn.second.second ) {
      toRtn.second.second = currentCoords[i].second;
      i = (start - 1);
    }
  } // for
  return(toRtn);
}

//=========================
// nextSymmetricDiffLine()
//=========================
TType nextSymmetricDiffLine(int numFiles) {
  /* Idea: Find minimum non-overlapped region between all files */
  static const unsigned long max = std::numeric_limits<long>::max();
  static const TType reset = std::make_pair(NOCHROM,
                                            std::make_pair(max, max));
  TType min = reset, next = reset;
  std::vector<long> allMins, allNext;
  for ( int i = 0; i < numFiles; ++i ) {
    if ( doneFile[i] || chrom[i] > min.first )
      continue;

    if ( chrom[i] < min.first ) {
      min.first = chrom[i];
      min.second = currentCoords[i];
      next = reset;
      allNext.clear();
      allMins.clear();
      allMins.push_back(i);
      continue;      
    }

    // chrom[i] == min.first
    if ( currentCoords[i].first < min.second.first ) {
      allNext.clear();
      next = min;
      allNext = allMins;
      allMins.clear();
      min.second = currentCoords[i];
      allMins.push_back(i);
    }
    else if ( currentCoords[i].first == min.second.first )
      allMins.push_back(i);
    else if ( currentCoords[i].first == next.second.first )
      allNext.push_back(i);
    else if ( currentCoords[i].first < next.second.first ) {
      allNext.clear();
      allNext.push_back(i);
      next.second = currentCoords[i];
    }
  } // for

  if ( allMins.empty() )
    return(NOGO);

  // Find minimum .second across allMins
  std::size_t minSize = allMins.size();
  unsigned long minSecond = max, minStart = max;
  for ( std::size_t x = 0; x < minSize; ++x ) {
    minStart = currentCoords[allMins[x]].first;
    if ( currentCoords[allMins[x]].second < minSecond )
      minSecond = currentCoords[allMins[x]].second;
  } // for
  unsigned long nextFirst = allNext.empty() ?
                                        max :
                                        currentCoords[allNext[0]].first;

  TType toRtn = NOGO;
  if ( minSize == 1 && allNext.empty() ) { // case 1
    toRtn.first = chrom[allMins[0]];
    toRtn.second = std::make_pair(minStart, minSecond);
    getNextFileCoords(allMins[0]);
  }
  else if ( allNext.empty() ) { // case 2
    for ( std::size_t x = 0; x < minSize; ++x ) {
      if ( minSecond == currentCoords[allMins[x]].second )
        getNextFileCoords(allMins[x]);
      else
        currentCoords[allMins[x]].first = minSecond + 1;
    } // for
    toRtn = CALLAGAIN;
    // Direct recursion removed to prevent any possible stack overflow
    // return(nextSymmetricDiffLine(numFiles)); // curse some more
  }
  else if ( minSize == 1 ) { // case 3
    toRtn.first = chrom[allMins[0]];
    if ( minSecond < nextFirst ) {
      toRtn.second = std::make_pair(minStart, minSecond);
      getNextFileCoords(allMins[0]);
    }
    else { // minSecond >= nextFirst && nextFirst >= 1
      toRtn.second = std::make_pair(minStart, nextFirst - 1);
      currentCoords[allMins[0]].first = nextFirst; // multiple new mins
    }
  }
  else { // multiple minimums, allNext nonempty: case 4
    if ( minSecond >= nextFirst ) { // all minimum overlap nextFirst
      for ( std::size_t x = 0; x < minSize; ++x )
        currentCoords[allMins[x]].first = nextFirst; // multiple mins
    }
    else { // minSecond < nextFirst;
      for ( std::size_t x = 0; x < minSize; ++x ) {
        if ( currentCoords[allMins[x]].second == minSecond )
          getNextFileCoords(allMins[x]);
        else
          currentCoords[allMins[x]].first = minSecond + 1;
      } // for
    }
    toRtn = CALLAGAIN;
    // Direct recursion removed to prevent any possible stack overflow
    // return(nextSymmetricDiffLine(numFiles)); // curse some more
  }
  return(toRtn);
}

//====================
// nextUnionAllLine()
//====================
TType nextUnionAllLine(int numFiles, int& marker) {
  /* Find next minimum entry between all files */
  PType inner = std::make_pair(std::numeric_limits<unsigned long>::max(), 
                               std::numeric_limits<unsigned long>::max());
  TType toRtn = std::make_pair(NOCHROM, inner);
  const int markSet = marker;
  for ( int i = 0; i < numFiles; ++i ) { // find minimum
    if ( !doneFile[i] ) {
      if ( chrom[i] < toRtn.first ) {
        toRtn.first = chrom[i];
        toRtn.second = currentCoords[i];
        marker = i;
      }
      else if ( chrom[i] == toRtn.first ) {
        if ( currentCoords[i].first < toRtn.second.first ) {
          toRtn.second = currentCoords[i];
          marker = i;
        }
        else if ( currentCoords[i].first == toRtn.second.first &&
                  currentCoords[i].second < toRtn.second.second ) {
          toRtn.second = currentCoords[i];
          marker = i;
        }
      }
    }
  } // for

  if ( marker == markSet )
    return(NOGO);
  return(toRtn);
}

} // namespace SetOperations
