// Author
//  Shane Neph : University of Washington

// Macro Guard
#ifndef INPUT_SETOPS_H
#define INPUT_SETOPS_H

// Files included
#include "Assertion.hpp"
#include "BedReader.hpp"
#include "SetOpsDefns.hpp"


namespace SetOperations {


//=======
// Input
//=======
struct Input {
  // Constructor
  Input(int argc, char** argv) {
    try {
      typedef InputError IE;
      typedef HelpException HE;
      Assert<HE>(argc > 1, "Help! - no mode detected");
      std::string mode = argv[1];
      Assert<IE>(mode.size() == 2, "Unknown mode: " + mode);
      bool startDash = (mode[0] == '-');
      bool nohelp = (mode[1] != 'h' && mode[1] != 'H');
      Assert<HE>(nohelp, "Help!");
      Assert<IE>(startDash, "No mode detected");
      Assert<IE>(argc > 2, "No input files specified");
      setModeType(mode[1]);
      numFiles_ = 0;
      current_ = 0;
      perc_ = 1;
      usePerc_ = true;
      int start = 2;
      if ( ft_ == ELEMENTOF || ft_ == NOTELEMENTOF ) { // extra args?
        int maxCount = 1, cntr = 0;
        for ( int i = 2; i < argc; ++i ) {
          std::string::size_type sz = std::string(argv[i]).size();
          if ( argv[i][0] == '-' && sz > 1 ) // sz of 1 means stdin
            setOption(argv[i]);
          else
            break;
          ++cntr;
          ++start;
        } // for
        std::string msg = "Cannot specify both % and bps";
        Assert<InputError>(cntr <= maxCount, msg);
      }

      bool onlyOne = true;
      for ( int i = start; i < argc; ++i ) {
        if ( std::string(argv[i]) == "-" ) {
          Assert<IE>(onlyOne, "Too many '-'");
          allFiles_.push_back(USESTDIN);
          onlyOne = !onlyOne;
        }
        else {
          std::ifstream check(argv[i]);
          Assert<IE>(check, "Cannot find " + std::string(argv[i]));
          allFiles_.push_back(argv[i]);
        }
        ++numFiles_;
      } // for
      Assert<InputError>(numFiles_ >= minFiles_, "Not enough files");
    } catch(HelpException& he) {
      HelpException toThrow(usage());
      throw(toThrow);
    } catch(InputError& ie) {
      std::string msg = usage();
      msg += "\n\nBad Input\n";
      msg += ie.GetMessage();
      InputError toThrow(msg);
      throw(toThrow);
    }
  }

  // Public methods
  ModeType GetModeType() const {
    return(ft_);
  }
  int NumberFiles() const {
    return(numFiles_);
  }
  BedReader* NextBed() {
    bool isOK = current_ < allFiles_.size();
    Assert<ProgramError>(isOK, "BedReader Creation Failure");
    if ( allFiles_[current_] != USESTDIN ) {
      /* streams_'s elements must persists until Input dies */
      streams_.push_back(new std::ifstream(allFiles_[current_].c_str()));
      std::istream_iterator<ByLine> k(*(streams_.back()));
      ++current_;
      return(new BedReader(k));
    }
    ++current_;
    std::istream_iterator<ByLine> k(std::cin);
    return(new BedReader(k));
  }
  double Threshold() const {
    return(perc_);
  }
  bool UsePercentage() const {
    return(usePerc_);
  }
  ~Input() {
    std::size_t sz = streams_.size();
    for ( std::size_t i = 0; i < sz; ++i )
      delete(streams_[i]);
  }

// private helpers
private:
  void setOption(const std::string& str) {
    typedef InputError IE;
    std::string l = str.substr(1); // get rid of -
    std::string::size_type pos = l.find("%");
    const std::string nums = ".1234567890";
    const std::string ints = "1234567890";
    if ( pos != std::string::npos ) {
      std::string value = l.substr(0, pos); // get rid of %
      Assert<IE>(!value.empty(), "Bad -% value");
      std::stringstream conv(value);
      Assert<IE>(value.find_first_not_of(nums) == std::string::npos,
                 "Bad: -% value");
      conv >> perc_;
      perc_ /= 100.0;
      if ( perc_ > 1 )
        perc_ = 1; // 100% is max
      usePerc_ = true;
      if ( perc_ == 0 ) { // 0% can match *everything*: convert to 1bp
        perc_ = 1;
        usePerc_ = false;
      }
    }
    else if ( l.find_first_not_of(ints) == std::string::npos ) {
      perc_ = atoi(l.c_str());
      usePerc_ = false; // pretend perc_ is # of bps
    }
    else
      throw(InputError("Unknown arg: " + str));
  }

  void setModeType(char t) {
    ModeType ft = MERGE;
    int min = 1;
    switch(t) {
      case 'c': case 'C':
        ft = COMPLEMENT; break;
      case 'd': case 'D':
        ft = DIFFERENCE; ++min; break;
      case 'e': case 'E':
        ft = ELEMENTOF; ++min; break;
      case 'i': case 'I':
        ft = INTERSECTION; ++min; break;
      case 'm': case 'M':
        ft = MERGE; break;
      case 'n': case 'N':
        ft = NOTELEMENTOF; ++min; break;
      case 's': case 'S':
        ft = SYMMETRIC_DIFFERENCE; ++min; break;
      case 'u': case 'U':
        ft = UNIONALL; ++min; break;
      default:
        InputError ie(std::string("Unknown set operation: ") + t);
        throw(ie);
    };
    ft_ = ft;
    minFiles_ = min;
  }

  std::string usage() const {
    std::string msg = "Only choose from one of:\n";
    msg += "-c File1 [File]*\n";
    msg += " [complement]\n";
    msg += "-d MasterFile File2 [File]*\n";
    msg += " [difference]\n";
    msg += "-e [-number% | -number (in bps)] RefFile File2 [File]*\n";
    msg += " [element of]\n";
    msg += "-i File1 File2 [File]*\n";
    msg += " [intersection]\n";
    msg += "-m File1 [File]*\n";
    msg += " [merge]\n";
    msg += "-n [-number% | -number (in bps)] RefFile File2 [File]*\n";
    msg += " [not element of]\n";
    msg += "-s File1 File2 [File]*\n";
    msg += " [symmetric difference]\n";
    msg += "-u File1 File2 [File]*\n";
    msg += " [union all]\n\n";
    msg += "NOTE: Use a dash (-) to indicate reading from stdin\n";
    msg += "NOTE: Only -e|n|u preserve all columns (no flattening)\n";
    msg += "NOTE: -e|n take longer to run than all other operations\n";
    return(msg);
  }

private:
  ModeType ft_;
  int numFiles_;
  int minFiles_;
  std::size_t current_;
  std::vector<std::string> allFiles_;
  std::vector<std::ifstream*> streams_;
  double perc_;
  bool usePerc_;
};


} // namespace SetOperations


#endif // INPUT_SETOPS_H
