// Macro Guard
#ifndef BEDMERGESORT_INPUT_H
#define BEDMERGESORT_INPUT_H

// Files included
#include "Assertion.hpp"
#include "Exception.hpp"
#include "StandardFiles.hpp"


//=======
// Input
//=======
struct Input {
  typedef Exception<-1> InputError;
  Input(int argc, char** argv, const std::string& prefix) : 
                            maxLines_(100000), maxFiles_(10),
                            in_(""), finalFile_("result.bed"),
                            verbose_(true) {
    std::string numErr = "expect positive numeric argument ";
    Assert<InputError>(argc >= 2, "no <input file>");
    std::string numbers = "0123456789";
    long cnt = 0;
    while ( ++cnt < argc-1 ) {
      std::string next = lower(argv[cnt]);
      Assert<InputError>(++cnt != argc-1, "did not find value: " + next);
      std::string value = argv[cnt];
      if ( next == "-maxlinesperfile" ) {
        Assert<InputError>(value.find_first_not_of(numbers), numErr + next);
        maxLines_ = atol(value.c_str());
        std::string errMsg = "No enough -maxlines; min=10,000";
        Assert<InputError>(maxLines_ >= 10000, errMsg);
      }
      else if ( next == "-maxopenfiles" ) {
        Assert<InputError>(value.find_first_not_of(numbers), numErr + next);
        maxFiles_ = atol(value.c_str());
        Assert<InputError>(maxFiles_ > 1, "Not enough -maxfiles");
      }
      else if ( next == "-resultsfile" )
        finalFile_ = value;
      else if ( next == "-verbose" )
        verbose_ = (value != "0");
      else
        throw(InputError("Unknown option: " + next));
    } // while
    in_ = argv[argc-1];
    std::ifstream inFile(argv[argc-1]);
    Assert<InputError>(inFile, "can't find: " + std::string(argv[argc-1]));
  }

  std::string FinalFile() const {
    return(finalFile_);
  }

  std::string InputFile() const {
    return(in_);
  }

  long MaxFiles() const {
    return(maxFiles_);
  }

  long MaxLines() const {
    return(maxLines_);
  }

  static std::string Usage() {
    std::string msg = "\nbedmergesort [";
    msg += " -maxLinesPerFile 100000";
    msg += " -maxOpenFiles 10";
    msg += " -resultsFile result.bed";
    msg += " -verbose 1";
    msg += " ] <inputfile>";
    return(msg);
  }

  bool Verbose() const {
    return(verbose_);
  }

private:
  std::string lower(const std::string& s) {
    std::string toRtn(s);
    std::string::size_type sz = s.size();
    for ( std::string::size_type i = 0; i < sz; ++i )
      toRtn[i] = std::tolower(toRtn[i]);
    return(toRtn);
  }

private:
  long maxLines_;
  long maxFiles_;
  std::string in_;
  std::string finalFile_;
  bool verbose_;
};

#endif // BEDMERGESORT_INPUT_H
